-- You can run, and re-run, this script any number of times.
-- You need only to be able to start by logging on as "system" (or similar)
-- and to know that you can drop and re-create users called "stopwatch_owner"
-- and "client" without harming anybody else.
--
-- I tested it in Oracle Database 18c Version 18.4.0.0.0.
-- It should run in any currently available Oracle Database version.
--------------------------------------------------------------------------------

SET TIMING OFF
CLEAR SCREEN
CONNECT sys/oracle@x/xe AS SYSDBA
alter user system identified by p container=all
/

-- START HERE.
--------------------------------------------------------------------------------

CONNECT system/p@x/r18

declare
  user_doesnt_exist exception;
  pragma exception_init(user_doesnt_exist, -01918);
  public_synonym_doesnt_exist exception;
  pragma exception_init(public_synonym_doesnt_exist, -01432);
begin
  begin
    execute immediate 'drop user stopwatch_owner cascade';
  exception when user_doesnt_exist then null; end;

  begin
    execute immediate 'drop user client cascade';
  exception when user_doesnt_exist then null; end;

  begin
    execute immediate 'drop public synonym stopwatch';
  exception when public_synonym_doesnt_exist then null; end;
end;
/
grant
  create session,
  create procedure
to stopwatch_owner identified by p
/
-- The target of the synonym doesn't have to exist yet.
create public synonym stopwatch for stopwatch_owner.stopwatch
/
grant
  create session,
  create procedure
to client identified by p
/
--------------------------------------------------------------------------------

CONNECT stopwatch_owner/p@x/r18

-- Turn on within-body subprogram inlining.
alter session set Plsql_Optimize_Level = 3
/
-- Suppress the warnings that report that inlining was done
-- and that inlined subprograms themselves were removed from the compiled code.
alter session set Plsql_Warnings = 'Error:All, Disable:6005, Disable:6006'
/
create package stopwatch_owner.stopwatch
  authid definer
is
  procedure  start_;
  function   reading return varchar2;
end stopwatch;
/
create package body stopwatch_owner.stopwatch
is
  start_moment double precision not null := 0;

  function duration_as_text(t in number) return varchar2 is
    ms_pr_sec         constant number not null := 1000.0;
    secs_pr_min       constant number not null := 60.0;
    mins_pr_hour      constant number not null := 60.0;
    secs_pr_hour      constant number not null := mins_pr_hour*secs_pr_min;
    secs_pr_day       constant number not null := 24.0*secs_pr_hour;

    confidence_limit  constant number not null := 0.02;
    ms_limit          constant number not null := 5.0;
    cs_limit          constant number not null := 10.0;

    result_           varchar2(32767) not null := '?';

    function fmt(n in number, template in varchar2) return varchar2 is
    begin
      return ltrim(to_char(n, template));
    end;
  begin
    case
      when t < confidence_limit then
        result_ := 'less than ~20 ms';

      when t >= confidence_limit and t < ms_limit then
        result_ := fmt(t*ms_pr_sec, '9999')||' ms';

      when t >= ms_limit and t < cs_limit then
        result_ := fmt(t, '90.99')||' ss';

      when t >= cs_limit and t < secs_pr_min then
        result_ := fmt(t, '99.9')||' ss';

      when t >= secs_pr_min and t < secs_pr_hour then
        declare
          ss   constant number not null := round(t);
          mins constant int     not null := trunc(ss/secs_pr_min);
          secs constant int     not null := ss - mins*secs_pr_min;
        begin
          result_ := fmt(mins, '09')||':'||fmt(secs, '09')||' mi:ss';
        end;

      when t >= secs_pr_hour and t < secs_pr_day then
        declare
          mi    constant number not null := round(t/secs_pr_min);
          hours constant int     not null := trunc(mi/mins_pr_hour);
          mins  constant int     not null := round(mi - hours*mins_pr_hour);
        begin
          result_ := fmt(hours, '09')||':'||fmt(mins,  '09')||' hh:mi';
        end;

      when t >= secs_pr_day then
        declare
          days  constant int     not null := trunc(t/secs_pr_day);
          mi    constant number not null := (t - days*secs_pr_day)/secs_pr_min;
          hours constant int     not null := trunc(mi/mins_pr_hour);
          mins  constant int     not null := round(mi - hours*mins_pr_hour);
        begin
          result_ := fmt(days,  '99')||' days '||
                    fmt(hours, '09')||':'||fmt(mins,  '09')||' hh:mi';
        end;
    end case;
    return result_;
  end duration_as_text;

  procedure start_ is
  begin
    start_moment := dbms_utility.get_time()/100.0;
  end start_;

  function reading return varchar2 is
  begin
    return duration_as_text(dbms_utility.get_time()/100.0 - start_moment);
  end reading;

begin
  -- Optional package intialization code.
  null;
end stopwatch;
/
grant execute on stopwatch_owner.stopwatch to client
/
--------------------------------------------------------------------------------

CONNECT stopwatch_owner/p@x/r18
alter session set Plsql_Optimize_Level = 3
/
alter session set Plsql_Optimize_Level = 3
/
create procedure stopwatch_test(t in number)
  authid definer
is
begin
  stopwatch.start_();
  dbms_session.sleep(t);
  dbms_output.put_line('stopwatch.reading(): '||stopwatch.reading());
end;
/
-- Warm up
begin
  stopwatch_test(0);
end;
/

-- Real test
begin
  stopwatch_test(1.234);
  stopwatch_test(5.678);
end;
/
