-- You can run, and re-run, this script any number of times.
-- You need only to be able to start by logging on as a superuser
-- (I used my own, called "super") in any suitable database (mine is called "demo").
-- And you need to know that you can drop and re-create
-- schemas called "stopwatch" and "stopwatch_body" and users called
-- "stopwatch_owner" and "client" without harming anybody else.
--
-- I tested it in Version 14.1 and (indirectly using YugabyteDB) in Version 11.2.
-- It should run in any currently available PostgreSQL version.
--------------------------------------------------------------------------------

\c demo super
set client_min_messages = warning;

-- User "stopwatch_owner"
do $body$
begin
  drop owned by stopwatch_owner cascade;
exception
  when undefined_object then null;
end;
$body$;
drop user if exists stopwatch_owner;
create user stopwatch_owner login password 'p';
grant create on database demo to stopwatch_owner;
create schema stopwatch      authorization stopwatch_owner; -- stopwatch spec
create schema stopwatch_body authorization stopwatch_owner; -- the body
alter user stopwatch_owner set search_path = stopwatch, stopwatch_body;

-- User "client"
do $body$
begin
  drop owned by client cascade;
exception
  when undefined_object then null;
end;
$body$;
drop user if exists client;
create user client login password 'p';
grant usage on schema stopwatch to client;
create schema client authorization client;
alter user client set search_path = client, stopwatch;

--------------------------------------------------------------------------------

\c demo stopwatch_owner
-- select current_user;
set client_min_messages = warning;

-- BODY
create function stopwatch_body.fmt(n in numeric, template in text)
  returns text
  stable
  security definer
  language plpgsql
as $body$
begin
  return ltrim(to_char(n, template));
end;
$body$;

create function stopwatch_body.duration_as_text(t in numeric)
  returns text
  stable
  security definer
  language plpgsql
as $body$
declare
  ms_pr_sec         constant numeric not null := 1000.0;
  secs_pr_min       constant numeric not null := 60.0;
  mins_pr_hour      constant numeric not null := 60.0;
  secs_pr_hour      constant numeric not null := mins_pr_hour*secs_pr_min;
  secs_pr_day       constant numeric not null := 24.0*secs_pr_hour;

  confidence_limit  constant numeric not null := 0.02;
  ms_limit          constant numeric not null := 5.0;
  cs_limit          constant numeric not null := 10.0;

  result                     text    not null := '';
begin
  case
    when t < confidence_limit then
      result := 'less than ~20 ms';

    when t >= confidence_limit and t < ms_limit then
      result := stopwatch_body.fmt(t*ms_pr_sec, '9999')||' ms';

    when t >= ms_limit and t < cs_limit then
      result := stopwatch_body.fmt(t, '90.99')||' ss';

    when t >= cs_limit and t < secs_pr_min then
      result := stopwatch_body.fmt(t, '99.9')||' ss';

    when t >= secs_pr_min and t < secs_pr_hour then
      declare
        ss   constant numeric not null := round(t);
        mins constant int     not null := trunc(ss/secs_pr_min);
        secs constant int     not null := ss - mins*secs_pr_min;
      begin
        result := stopwatch_body.fmt(mins, '09')||':'||stopwatch_body.fmt(secs, '09')||' mi:ss';
      end;

    when t >= secs_pr_hour and t < secs_pr_day then
      declare
        mi    constant numeric not null := round(t/secs_pr_min);
        hours constant int     not null := trunc(mi/mins_pr_hour);
        mins  constant int     not null := round(mi - hours*mins_pr_hour);
      begin
        result := stopwatch_body.fmt(hours, '09')||':'||stopwatch_body.fmt(mins,  '09')||' hh:mi';
      end;

    when t >= secs_pr_day then
      declare
        days  constant int     not null := trunc(t/secs_pr_day);
        mi    constant numeric not null := (t - days*secs_pr_day)/secs_pr_min;
        hours constant int     not null := trunc(mi/mins_pr_hour);
        mins  constant int     not null := round(mi - hours*mins_pr_hour);
      begin
        result := stopwatch_body.fmt(days,  '99')||' days '||
                  stopwatch_body.fmt(hours, '09')||':'||stopwatch_body.fmt(mins,  '09')||' hh:mi';
      end;
  end case;
  return result;
end;
$body$;

create procedure stopwatch_body.start()
  security definer
  language plpgsql
as $body$
declare
  -- Make a memo of the current wall-clock time as
  -- seconds since midnight on 1-Jan-1970.
  start_moment constant text not null := extract(epoch from clock_timestamp())::text;
begin
  execute format('set stopwatch.start_moment to %L', start_moment);
end;
$body$;

create function stopwatch_body.reading()
  returns text
  volatile
  security definer
  language plpgsql
as $body$
declare
  start_moment  constant double precision not null := current_setting('stopwatch.start_moment');
  curr_moment   constant double precision not null := extract(epoch from clock_timestamp());
begin
  return stopwatch_body.duration_as_text((curr_moment - start_moment)::numeric);
end;
$body$;

-- SPEC
create procedure stopwatch.start()
  security definer
  language plpgsql
as $body$
begin
  call stopwatch_body.start();
end;
$body$;
grant execute on procedure stopwatch.start() to client;

create function stopwatch.reading()
  returns text
  volatile
  security definer
  language plpgsql
as $body$
begin
  return stopwatch_body.reading();
end;
$body$;
grant execute on function stopwatch.reading() to client;

--------------------------------------------------------------------------------

\c demo client
-- select current_user;
set client_min_messages = warning;

-- \sf stopwatch.reading()

do $body$
declare
  t text;
begin
  call stopwatch_body.start();
  assert false, 'Unexpected';
exception
  -- 42501: permission denied for schema stopwatch_body
  when insufficient_privilege then null;
end;
$body$;

create function client.stopwatch_test(t in double precision)
  returns table(z text)
  security definer
  volatile
  language plpgsql
as $body$
begin
  call stopwatch.start();
  perform pg_sleep(t);
  z := stopwatch.reading(); return next;
end;
$body$;

-- Warm up
select z from client.stopwatch_test(0);

-- Real test
select z from client.stopwatch_test(3.456);
select z from client.stopwatch_test(5.678);
