#!/bin/bash

# To setup the publisher and execute SQL file.

# The 1st parameter is which streaming mode to test, 'parallel' or 'on'. For
# 'on' mode, test based on HEAD; for 'parallel' mode, test based on patched
# code.

# The 2nd parameter is logical_decoding_work_mem specified in the test.

# The 3rd parameter is which rollback case to test. It determines which SQL file
# to use. "1" means rollback 10%, "2" means rollback 20%, "3" means rollback
# 30%, "5" means rollback 50%.

# The SQL files need to placed in a directory with this file.
#
# SQL files:
#  rollback_1.sql - SQL for rollback 10% case.
#  rollback_2.sql - SQL for rollback 20% case.
#  rollback_3.sql - SQL for rollback 30% case.
#  rollback_5.sql - SQL for rollback 50% case.

# If you want to run this script, you might need to change the install path in
# the scripts:
#  path_parallel_apply : install path of patched code.
#  path_master : install path of HEAD.

# There are two servers in the test, one is the publisher and the other is the
# subscriber. This test uses synchronous logical replication,
# "synchronous_standby_names" is set after the subscription is created, the
# publisher waits for the subscriber to be created for 20s in the script, you
# might want to change the waiting time to ensure that the subscriber has been
# created.

# The number of runs can be changed by modifying the value of `loop`.

###############################################################################
# Steps to test:
# =========================
# 1. Execute `start_pub.sh` on publisher.
# 2. After the publication is created (see the output of publisher script: 'wait
#    for subscriber...' ), execute `start_sub.sh` on subscriber.
# 3. Wait for `start_pub.sh` to finish executing. The time of each run will be
#    output.
# 
# Example 1:
# To test 'parallel' mode, logical_decoding_work_mem=64kB, rollback
# 10% case.
# On publisher, execute `sh start_pub.sh parallel 64kB 1`.
# On subscriber, execute `sh start_sub.sh parallel`.
# 
# Example 2:
# To test 'on' mode, logical_decoding_work_mem=64MB, rollback 20%
# case.
# On publisher, execute `sh start_pub.sh on 64MB 2`.
# On subscriber, execute `sh start_sub.sh on`.
# 
# After all the tests, you need to stop the pg server manually. On publisher,
# execute `pg_ctl -D data_pub stop`. On subscriber, execute `pg_ctl -D data_sub
# stop`.
###############################################################################


path_master=/home/postgres/install_head/bin/
path_parallel_apply=/home/postgres/install_pa/bin/

loop=10

if [ -n "$3" ]; then
    echo $1 $2 $3
else
    echo this script needs 3 parameters
    exit 0
fi

if [ "$1" = "on" ]; then
    path=$path_master
elif [ "$1" = "parallel" ]; then
    path=$path_parallel_apply
fi

pub_port=5432

${path}pg_ctl -D data_pub stop -m i
rm -rf data_pub pub.log

${path}initdb -D data_pub -U postgres

cat << EOF >> data_pub/postgresql.conf
listen_addresses = '*'
wal_level = logical
port = $pub_port
logical_decoding_work_mem = $2
shared_buffers = 8GB
checkpoint_timeout = 30min
max_wal_size = 20GB
min_wal_size = 10GB
autovacuum = off
EOF

cat << EOF >> data_pub/pg_hba.conf
host    all             all             all            trust
host    replication     all             all            trust
EOF

${path}pg_ctl -D data_pub start -l pub.log

${path}psql -d postgres -U postgres -p $pub_port -c "CREATE TABLE large_test (\
     id int primary key,\
     num1 bigint,\
     num2 double precision,\
     num3 double precision\
);"
${path}psql -d postgres -U postgres -p $pub_port -c "create publication pub for table large_test;"

echo "==============================================="
echo wait for subscriber...
sleep 20

echo set sync replication
echo synchronous_standby_names = \'any 1\(sub\)\' >> data_pub/postgresql.conf
${path}pg_ctl -D data_pub reload

echo 10kk rollback ${3}0%  work_mem=$2 ${path} >res.txt
for i in `seq 1 $loop`; do
    ${path}psql -d postgres -U postgres -p $pub_port -f rollback_${3}.sql | tee -a res.txt
done

echo "===============================================" >> res_${3}0.txt
echo  ${path} $2 rollback ${3}0% | tee -a res_${3}0.txt

grep '^Time' res.txt | awk '{print $2}' | awk '{
    if (NR%22 == 1)
        sum = $1;
    else
        sum += $1;
    if (NR%22 == 0)
        print sum/1000;
}'  | tee -a res_${3}0.txt
