#!/bin/bash

# Test configuration

# Test uses configuration with three DB instances
#  A - primary instance
#  B - hot standby for A, uses physical replication
#  C - downstream server for B
# By default this script tries to cleanup previously
# created database instances and then create required
# configuration.
# Note: THIS SCRIPT DELETES directories specified in 
# 'pg_data' map.
# To perform only cleanup of existing instances use
# option '--cleanup'

# Path to installed PG binaries
PG_PATH=/opt/pgpro/pg-master
# Location of root data and logs directories
PG_DATA=data
PG_LOGS=logs

# Listen port for each databae instance
declare -A pg_port=(
                    A 5432 
                    B 5433 
                    C 5434
                   )
# Location to install database instances
declare -A pg_data=(
                    A "$PG_DATA/data-A" 
                    B "$PG_DATA/data-B" 
                    C "$PG_DATA/data-C"
                   )
# Location of log files for each database instance
declare -A pg_logs=(
                    A "$PG_LOGS/log-A.log" 
                    B "$PG_LOGS/log-B.log" 
                    C "$PG_LOGS/log-C.log"
                   )

# Exit on first error
set -e

echo "Stop existing servers (if any)"
for pg in A B C
do
  export PGPORT=${pg_port[$pg]}
  export PGDATA=${pg_data[$pg]}
  [ -f $PGDATA/postmaster.pid ] && $PG_PATH/bin/pg_ctl stop
done

echo "Remove existing servers (if any)"
for pg in A B C
do
  [ -d ${pg_data[$pg]} ] && rm -rf ${pg_data[$pg]}
  [ -f $PG_LOGS/log-$pg.log ] && rm $PG_LOGS/log-$pg.log
done

if [ "$1" == "--cleanup" ]; then
  echo "Cleanup completed"
  exit 0
fi

# Create required directories
[ ! -d "$PG_DATA" ] && mkdir -p "$PG_DATA"
[ ! -d "$PG_LOGS" ] && mkdir -p "$PG_LOGS"

echo "Create new primary server (A)"
$PG_PATH/bin/initdb --data-checksums --pgdata=${pg_data[A]}
echo "port = ${pg_port[A]}" >> ${pg_data[A]}/postgresql.auto.conf
echo "wal_level = 'logical'" >> ${pg_data[A]}/postgresql.auto.conf
echo "max_wal_senders = 200" >> ${pg_data[A]}/postgresql.auto.conf
echo "max_replication_slots = 200" >> ${pg_data[A]}/postgresql.auto.conf
echo "max_worker_processes = 500" >> ${pg_data[A]}/postgresql.auto.conf
echo "max_active_replication_origins = 200" >> ${pg_data[A]}/postgresql.auto.conf
$PG_PATH/bin/pg_ctl --pgdata=${pg_data[A]} --log=${pg_logs[A]} start

echo "Generate test tables with data on instance A"
echo "select 'create table repli_test_t'||gt||'(id numeric,'||string_agg('c'||gc||' numeric', ',')||', constraint repli_test_t'||gt||'_pk primary key(id)) with (fillfactor=70)' from generate_series(0,9) gc, generate_series(0,99) gt group by gt \gexec" | $PG_PATH/bin/psql --port=${pg_port[A]}
echo "select 'insert into repli_test_t'||gt||'(id,'||string_agg('c'||gc, ',')||') select g::bigint,'||string_agg('random()*100e3::bigint', ',')||' from generate_series(0,10e3) g' from generate_series(0,9) gc, generate_series(0,99) gt group by gt \gexec" | $PG_PATH/bin/psql --port=${pg_port[A]}
$PG_PATH/bin/psql --port=${pg_port[A]} --command="checkpoint"

echo "Create physical standby instance (B)"
$PG_PATH/bin/pg_basebackup --port=${pg_port[A]} --pgdata=${pg_data[B]} --write-recovery-conf --slot=data_b --create-slot
sed -i "s/port = ${pg_port[A]}/port = ${pg_port[B]}/" ${pg_data[B]}/postgresql.auto.conf
echo "hot_standby = on" >> ${pg_data[B]}/postgresql.auto.conf
echo "hot_standby_feedback = on" >> ${pg_data[B]}/postgresql.auto.conf
$PG_PATH/bin/pg_ctl --pgdata=${pg_data[B]} --log=${pg_logs[B]} start

echo "Create instance for logical replication (C)"
$PG_PATH/bin/pg_basebackup --port=${pg_port[A]} --pgdata=${pg_data[C]}
sed -i "s/port = ${pg_port[A]}/port = ${pg_port[C]}/" ${pg_data[C]}/postgresql.auto.conf
echo "max_logical_replication_workers = 200" >> ${pg_data[C]}/postgresql.auto.conf
$PG_PATH/bin/pg_ctl --pgdata=${pg_data[C]} --log=${pg_logs[C]} start

echo "Create publications for each table on instance A"
echo "select 'create publication pub_'||gt||' for table repli_test_t'||gt from generate_series(0,99) gt \gexec" | $PG_PATH/bin/psql --port=${pg_port[A]} 
$PG_PATH/bin/psql --port=${pg_port[A]} --command="select pg_log_standby_snapshot()"
$PG_PATH/bin/psql --port=${pg_port[A]} --command="checkpoint"
sleep 1

echo "Create logical replication slots on instance B"
$PG_PATH/bin/psql --port=${pg_port[B]} --command="select pg_create_logical_replication_slot('sub_0','pgoutput')" &
sleep 1
$PG_PATH/bin/psql --port=${pg_port[A]} --command="select pg_log_standby_snapshot()"
sleep 5
echo "select 'select pg_copy_logical_replication_slot(''sub_0'', ''sub_'||gt||''')' from generate_series(1,99) gt \gexec" | $PG_PATH/bin/psql --port=${pg_port[B]}
sleep 5

echo "Create subscriptions on instance C"
echo "select 'create subscription sub_'||gt||' connection ''port=${pg_port[B]}'' publication pub_'||gt||' with (create_slot=false, copy_data=false)' from generate_series(0,99) gt \gexec" | $PG_PATH/bin/psql --port=${pg_port[C]}

$PG_PATH/bin/psql --port=${pg_port[A]} --command="select pg_log_standby_snapshot()"
$PG_PATH/bin/psql --port=${pg_port[A]} --command="checkpoint"

echo "All done"