#!/bin/bash

set -euo pipefail

CONTAINER_NAME=pg_main
DATA_DIR="pgdata"
BACKUP_DIR="backup"
PADDING="................................................."

# Returns curent LSN
get_current_lsn() {
    docker exec -i "$1" psql -U postgres -t -A -c "select pg_current_wal_lsn();"
}

# Prints a delta between two LSN in bytes.
get_lsn_delta(){
    local prev=$1
    local next=$2

    echo $(( 0x${next: -8} - 0x${prev: -8} ))
}

print_current_lsn(){
    print_padded "Current LSN: $(get_current_lsn "$1")"
}

# Since Postgres needs some time to create WAL summaries required for
# incremental backups, we need to wait until this summaries are generated.
# Because this process cannot be controlled directly, this script waits until
# there is no active WAL writing.
wait_until_wal_idle() {
    local seconds_idle=0
    local container_name=$1
    local cur_lsn
    local status
    cur_lsn=$(get_current_lsn "$container_name")
    println_padded "Waiting WAL stabilization, current LSN:" "$cur_lsn"
    while [ $seconds_idle -lt 30 ] 
    do
        local next_lsn
        next_lsn=$(get_current_lsn "$container_name")
        local delta
        delta=$(get_lsn_delta "$cur_lsn" "$next_lsn")
        if [[ $delta -lt 100 ]]; then
            seconds_idle=$(( seconds_idle+1 ))
            status=", waiting $((30-seconds_idle)) sec."
        else
            seconds_idle=0
            status=""
        fi
        printf "\r"
        msg="Current WAL file activity (byte/s)"
        printf "%s%s%-25s" "$msg" "${PADDING:${#msg}}" "$delta$status"
        cur_lsn=$next_lsn

        sleep 1
    done
    printf "\n"
    println_padded "WAL stabilized at" "$cur_lsn"
}


restart_postgres(){
    print_padded "Restarting PostgreSQL container"
    docker compose -f docker-compose.yml stop > /dev/null 2>&1
    docker compose -f docker-compose.yml up -d > /dev/null 2>&1
    print_ok

    print_padded "Waiting for connection"
    until docker exec "$CONTAINER_NAME" pg_isready -q; do
        sleep 1
    done
    sleep 3
    print_ok
}

print_padded(){
    msg=$1
    suffix=${2-}
    printf "%s%s%s" "$msg" "${PADDING:${#msg}}" "$suffix"
}

println_padded(){
    msg=$1
    suffix=${2-}
    printf "%s%s%s\n" "$msg" "${PADDING:${#msg}}" "$suffix"
}

print_ok(){
  printf "\e[32m%s\e[0m\n" "OK"
}

checkpoint(){
    print_padded "Perform Checkpoint"
    docker exec -i "$CONTAINER_NAME" psql -U postgres -c "checkpoint;" >/dev/null 2>&1
    print_ok
}

# Step 1. 
# Checking presence of necessary directories
for d in "$DATA_DIR" "$BACKUP_DIR"; do
    print_padded "Checking directory $d"
    if [ -d "$d" ]; then
        rm -rf "$d"
    fi
    mkdir -p "$d"
    print_ok
done

restart_postgres

# Step 2.
# Insert as many rows as it necessary to create sequencial TOAST relation file
QUERY_CALLS=30
ROWS_PER_TX=10000

# Insert 1000 rows into 'dummmy' table each row has 8K of random data
INSERT_8K_QUERY="INSERT INTO dummy(data)
                 SELECT (
                    SELECT string_agg(gen_random_bytes(1000), '' ORDER BY n)
                    FROM generate_series(1, 8) AS g(n)
                 )
                 FROM generate_series(1, $ROWS_PER_TX) AS s(i);"

for (( i=1; i<=QUERY_CALLS; i++ )); do
  if docker exec -i "$CONTAINER_NAME" psql -U postgres -c "$INSERT_8K_QUERY" > /dev/null 2>&1; then
      printf "\r"
      print_padded "Generating $((i*ROWS_PER_TX))/$((QUERY_CALLS*ROWS_PER_TX)) rows"
  fi
done
print_ok

# Step 3.
# Let PostgreSQL finish some internal work to generate WAL summaries
wait_until_wal_idle "$CONTAINER_NAME"

# Step 4.
# Creating Base Backup
printf "%s\n" "======= Starting Base Backup ======"

checkpoint
wait_until_wal_idle "$CONTAINER_NAME"

print_padded "Performing Base Backup"
docker exec -i "$CONTAINER_NAME" bash -c "pg_basebackup -U postgres -D /var/lib/postgresql/backup/base_backup -Fp -P" >/dev/null 2>&1
print_ok

print_padded "Deleting some rows for future truncation"
docker exec -i "$CONTAINER_NAME" psql -U postgres -c "delete from dummy where id > 290000;" >/dev/null 2>&1
print_ok

checkpoint
wait_until_wal_idle "$CONTAINER_NAME"

printf "%s\n" "Running vacuum"

docker exec -i "$CONTAINER_NAME" psql -U postgres -c "vacuum (verbose, truncate) dummy;"

checkpoint
wait_until_wal_idle "$CONTAINER_NAME"

printf "%s\n" "======= Starting Incremental Backup ======"

print_padded "Performing Incremental Backup"
docker exec -i "$CONTAINER_NAME" pg_basebackup -U postgres --incremental=/var/lib/postgresql/backup/base_backup/backup_manifest -D /var/lib/postgresql/backup/incremental_backup -Fp -P
print_ok

printf  "%s\n" "======= Starting Combine Backup ======"

docker exec -i "$CONTAINER_NAME" pg_combinebackup /var/lib/postgresql/backup/base_backup/ /var/lib/postgresql/backup/incremental_backup -o /var/lib/postgresql/backup/restored

printf "%s\n" "Script finished. At this point, an error should have been reported above, indicating that pg_combinebackup failed to restore the backup."
